/******************************************************************************
 * vtovrscl.c - Chapter 8 sample code                                         *
 *                                                                            *
 * To be used with mach64VT/3D RAGE sample code.                              *
 * This module contains routines to manipulate the VT overlay/scaler.         *
 *                                                                            *
 * Copyright (c) 1994-1998 ATI Technologies Inc.  All rights reserved.        *
 ******************************************************************************/

//#include <i86.h>
#include "atim64vt.h"
#include "definevt.h"
#include "main.h"


/******************************************************************************
 * set_overlay                                                                *
 *  Function: uses the ATI264VT to set the overlay window.                    *
 *            The overlay window is placed at (x, y) of size (width x         *
 *            height) using the current settings.  For modes that use         *
 *            double-scan (typically low resolution modes like 320x200        *
 *            or 320x240) both the y-coordinate and the height must be        *
 *            multiplied by 2 to compensate for the double-scan.              *
 *    Inputs: x - starting x coordinate in pixels (left most)                 *
 *            y - starting y coordinate in pixels (top most)                  *
 *            width - width of overlay window in pixels                       *
 *            height - height of overlay window in pixels                     *
 *   Outputs: NONE                                                            *
 ******************************************************************************/

void set_overlay (int x, int y, int width, int height)
{
    // If a double-scan mode, double the y and height values.
    // Note: CRTC_GEN_CNTL is non-FIFOed, so wait_for_idle () not needed.

    if (regr (CRTC_GEN_CNTL) & 0x00000001)
    {
        y *= 2;
        height *= 2;
    } // if

    wait_for_fifo (2);

    regw (OVERLAY_Y_X, (x << 16) + y);
    regw (OVERLAY_Y_X_END, ((x + width) << 16) + y + height);

    return;

} // set_overlay


/******************************************************************************
 * set_scale                                                                  *
 *  Function: uses the ATI264VT to set the scaler size.                       *
 *            The overlay window is placed at (x, y) of size (width x         *
 *            height) using the current settings.  For modes that use         *
 *            double-scan (typically low resolution modes like 320x200        *
 *            or 320x240) both the y-coordinate and the height must be        *
 *            multiplied by 2 to compensate for the double-scan.              *
 *    Inputs: x - starting x coordinate in pixels (left most)                 *
 *            y - starting y coordinate in pixels (top most)                  *
 *            width - width of overlay window in pixels                       *
 *            height - height of overlay window in pixels                     *
 *   Outputs: NONE                                                            *
 ******************************************************************************/

void set_scale (int src_width, int src_height, int dst_width, int dst_height)
{
    long h_inc, v_inc;

    // If a double-scan mode, double the destination height value.
    // Note: CRTC_GEN_CNTL is non-FIFOed, so wait_for_idle () not needed.

    if (regr (CRTC_GEN_CNTL) & 0x00000001)
    {
        dst_height *= 2;
    } // if

    // Compute horizontal and vertical scaling values in
    // fixed point 4.12 binary format.

    h_inc = (src_width << 12) / dst_width;
    v_inc = (src_height << 12) / dst_height;

    wait_for_fifo (2);

    regw (OVERLAY_SCALE_INC, (h_inc << 16) + v_inc);
    regw (SCALER_HEIGHT_WIDTH, (src_width << 16) + src_height);

    return;

} // set_scale
