/******************************************************************************
 * Header file to mach64 Chapter 6 sample code                                *
 *                                                                            *
 * Includes structures, prototypes, and globals.                              *
 *                                                                            *
 * Copyright (c) 1994-1997 ATI Technologies Inc.  All rights reserved.        *
 ******************************************************************************/

#ifndef _MAIN_H
#define _MAIN_H

#include "defines.h"

// Structures.

typedef struct                          // See mach64 BIOS extension info.
{
    unsigned short size;
    unsigned char revision;
    unsigned char mode_tables;
    unsigned short mode_table_offset;
    unsigned char mode_table_size;
    unsigned char vga_type;             // See VGA types.
    unsigned short asic_id;
    unsigned char vga_boundary;
    unsigned char memory_size;
    unsigned char dac_type;             // See DAC types.
    unsigned char memory_type;
    unsigned char bus_type;             // See BUS types.
    unsigned char monitor_cntl;
    unsigned short aperture_addr;
    unsigned char aperture_cfg;
    unsigned char colour_depth_support;
    unsigned char ramdac_feature_support;
    unsigned char io_type;
    unsigned short reserved1;
    unsigned short io_base;
    unsigned char reserved2[6];
} query_structure;

typedef struct
{
    unsigned int io_base;
    unsigned int io_type;
} io_query;

typedef struct
{
    int xres;
    int yres;
    int bpp;
    int depth;                          // Colour depth support.
    int pitch;
    int vga_aperture_status;            // ON or OFF.
    int linear_aperture_status;         // ON or OFF.
    int linear_aperture_size;           // 4MB or 8MB.
    unsigned long linear_address;       // Physical linear aperture address.
    unsigned long virt_seg;             // Virtual linear aperture address.
    unsigned long vga_memreg_offset;    // VGA aperture memory register offset.
    unsigned long linear_memreg_offset; // Linear aperture memory register offset.
} modecfg;


typedef struct
{
    int red;
    int green;
    int blue;
} palette;

typedef struct
{
    int x;
    int y;
} point;

typedef struct
{
    int length;
    point *point_ptr;
} polygon;

typedef struct
{
    int y;
    int width;
    int height;
    int current_x;
    int current_y;
    int hot_x;
    int hot_y;
    unsigned long cur_offset;
    unsigned long colour0;
    unsigned long colour1;
    unsigned int bitmap[HWCURHEIGHT * HWCURWIDTH];
} hwcursor;

typedef enum {BLUE, GREEN, RED} primarycolour;


// Prototypes.

// Functions in memreg.c

int regw (unsigned int regindex, unsigned long data);
unsigned long regr (unsigned int regindex);

// Functions in ioreg.c

int iow32 (unsigned int regindex, unsigned long data);
unsigned long ior32 (unsigned int regindex);
int iow16 (int regindex, unsigned short data);
unsigned short ior16 (int regindex);
int iow8 (int regindex, char data);
char ior8 (int regindex);

// Functions in init.c

int init_graphics (void);
int get_old_mode (void);
int set_old_mode (void);
void process_command_line (int argc, char *argv[]);
void reset_engine (void);
void init_engine (void);
int start (int argc, char *argv[]);
int finish (void);

// Functions in m64bios.c

long phys_to_virt (long physical, long size);
int load_and_set_mode (int xres, int bpp, int pitch);
int enable_aperture (void);
int short_query_function (void);
int long_query (void);
int short_query (void);

// Functions in colour.c

unsigned long get_colour_code (int colour_index);
unsigned long get_primary_colour (primarycolour pri_col, unsigned long colour);
unsigned long get_xy_offset (int x, int y);

// Functions in palette.c

void set_palette (int index, palette entry);
palette get_palette (int index);
void save_palette (palette *palettebuffer);
void restore_palette (palette *palettebuffer);
void init_palette (void);
void init_palettized (void);

// Functions in wait.c

void wait_for_idle (void);
void wait_for_fifo (int entries);
void terminate (int idle_problem);

// Functions in draw.c

void draw_rectangle (int x, int y, int width, int height);
void clear_screen (int x, int y, int width, int height);
void draw_line (int x1, int y1, int x2, int y2);
void blit (int x1, int y1, int x2, int y2, int width, int height);

// Functions in hwcursor.c

void set_hwcursor (int y, int width, int height, int hot_x, int hot_y,
                   unsigned long colour0, unsigned long colour1,
                   unsigned int *bitmap);
void enable_hwcursor (void);
void disable_hwcursor (void);
void set_hwcursor_pos (int x, int y);
void get_hwcursor_pos (point *position);
void set_cursor_colours (unsigned long colour0, unsigned long colour1);


// Globals.

extern unsigned int IO_REGISTER[64];
extern modecfg MODE_INFO;
extern query_structure QUERY_DATA;
extern io_query IO_DATA;
extern int OLD_MODE;
extern int GMODE_RES;
extern int GCLR_DEPTH;
extern palette SAVEPALETTE[256];
extern hwcursor CURSORDATA;

#endif // _MAIN_H
