; ===========================================================================
; VINT.ASM
;
; Interface routines to handle MACH64 CRTC vertical interrupts.
;
; Compiling:
;   masm /Ml /D<memory model> romcalls.asm;
;       <memory model> = mem_S for SMALL model,
;                        mem_M for MEDIUM model,
;                        mem_L for LARGE model
;
; Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
; ===========================================================================

include ..\util\atim64.inc

IFDEF mem_S
PARM        equ     4   ; passed parameters start at bp+4 for small model
ELSE
PARM        equ     6   ; passed parameters start at bp+6 for other models
ENDIF

IFDEF mem_S
.MODEL  SMALL, C
ELSEIFDEF mem_M
.MODEL  MEDIUM, C
ELSE
.MODEL  LARGE, C
ENDIF

.DATA

.CODE
.286

IFDEF mem_S
extrn       ior8:NEAR
extrn       ior16:NEAR
extrn       ior32:NEAR
extrn       iow8:NEAR
extrn       iow16:NEAR
extrn       iow32:NEAR
ELSE
extrn       ior8:FAR
extrn       ior16:FAR
extrn       ior32:FAR
extrn       iow8:FAR
extrn       iow16:FAR
extrn       iow32:FAR
ENDIF

; Macro for 'call' model handling
Mcall       macro   routine
IFDEF mem_S
            call    NEAR PTR routine
ELSE
            call    FAR PTR routine
ENDIF
            endm

; macros to handle IO operations
Min         macro   inp_routine
            push    dx
            Mcall   inp_routine
            add     sp, 2
            endm

Mout        macro   outp_routine
            push    ax
            push    dx
            Mcall   outp_routine
            add     sp, 4
            endm

; ---------------------------------------------------------------------------
; ENABLE_VBLANKINT
;
; Enable the vertical blank interrupt bit.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  enable_vblankint

IFDEF mem_S
enable_vblankint proc near
ELSE
enable_vblankint proc far
ENDIF
            ; save registers used
            push    dx

            ; set vertical blank interrupt bit
            mov     dx, ioCRTC_INT_CNTL
            Min     ior8
            or      al, 2
            Mout    iow8

            ; restore saved registers
            pop     dx

            ret

enable_vblankint endp

; ---------------------------------------------------------------------------
; DISABLE_VBLANKINT
;
; Enable the vertical blank interrupt bit.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  disable_vblankint

IFDEF mem_S
disable_vblankint proc near
ELSE
disable_vblankint proc far
ENDIF
            ; save registers used
            push    dx

            ; clear vertical blank interrupt bit
            mov     dx, ioCRTC_INT_CNTL
            Min     ior8
            and     al, 0FDh
            Mout    iow8

            ; restore saved registers
            pop     dx

            ret

disable_vblankint endp

; ---------------------------------------------------------------------------
; WAIT_FOR_VBLANK
;
; Wait until vertical blank interrupt bit is set. It is assumed that the
; vertical blank interrupt enable bit is set.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  wait_for_vblank

IFDEF mem_S
wait_for_vblank proc near
ELSE
wait_for_vblank proc far
ENDIF
            ; save registers used
            push    dx

            ; ack vertical blank interrupt to clear it
            mov     dx, ioCRTC_INT_CNTL
            Min     ior8
            or      al, 6
            Mout    iow8

            ; wait until it gets set again
chkvblank:

            Min     ior8
            and     al, 4
            cmp     al, 4
            jne     chkvblank

            ; restore saved registers
            pop     dx

            ret

wait_for_vblank endp

; ---------------------------------------------------------------------------
; POLL_FOR_VBLANK
;
; Wait until vertical blank bit is set.  Effectively waits until the start of
; the next vertical blank period.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  poll_for_vblank

IFDEF mem_S
poll_for_vblank proc near
ELSE
poll_for_vblank proc far
ENDIF
            ; save registers used
            push    dx

            ; wait until bit clears
            mov     dx, ioCRTC_INT_CNTL
chkvblank1:
            Min     ior8
            test    al, 1
            jnz     chkvblank1

            ; now we are at the top of the display
            ; wait until bit gets set again
chkvblank2:
            Min     ior8
            test    al, 1
            jz      chkvblank2

            ; now we are at the start of the vblank region
            ; restore saved registers
            pop     dx

            ret

poll_for_vblank endp

; ---------------------------------------------------------------------------
; SET_VLINE
;
; Inputs : WORD vertical line to set interrupt bit at.
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  set_vline

IFDEF mem_S
set_vline   proc    near
ELSE
set_vline   proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save registers used
            push    dx

            ; set vertical line to set interrupt bit at
            mov     ax, WORD PTR [bp+PARM]
            mov     dx, ioCRTC_VLINE_CRNT_VLINE
            Mout    iow16

            ; restore saved registers
            pop     dx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

set_vline   endp

; ---------------------------------------------------------------------------
; ENABLEVLINEINT
;
; Enable the vertical line interrupt bit.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  enable_vlineint

IFDEF mem_S
enable_vlineint proc near
ELSE
enable_vlineint proc far
ENDIF
            ; save registers used
            push    dx

            ; set vertical line interrupt bit
            mov     dx, ioCRTC_INT_CNTL
            Min     ior8
            or      al, 8
            Mout    iow8

            ; restore saved registers
            pop     dx

            ret

enable_vlineint endp

; ---------------------------------------------------------------------------
; DISABLEVLINEINT
;
; Enable the vertical blank interrupt bit.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  disable_vlineint

IFDEF mem_S
disable_vlineint proc near
ELSE
disable_vlineint proc far
ENDIF
            ; save registers used
            push    dx

            ; clear vertical line interrupt bit
            mov     dx, ioCRTC_INT_CNTL
            Min     ior8
            and     al, 0F7h
            Mout    iow8

            ; restore saved registers
            pop     dx

            ret

disable_vlineint endp

; ---------------------------------------------------------------------------
; WAIT_FOR_VLINE
;
; Wait until vertical line interrupt bit is set. It is assumed that the
; vertical line interrupt enable bit is set. Also, it is assumed that the
; CRTC_VLINE vertical line has been previously set in the
; CRTC_VLINE_CRNT_VLINE register.
;
; Inputs : none
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  wait_for_vline

IFDEF mem_S
wait_for_vline proc near
ELSE
wait_for_vline proc far
ENDIF
            ; save registers used
            push    dx

            ; ack vertical line interrupt to clear it
            mov     dx, ioCRTC_INT_CNTL
            Min     ior8
            or      al, 18h
            Mout    iow8

            ; wait until it gets set again
chkvline:
            Min     ior8
            and     al, 10h
            cmp     al, 10h
            jne     chkvline

            ; restore saved registers
            pop     dx

            ret

wait_for_vline endp

            end

